function TumorGrowthMove
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%            MBI Bootcamp in Cancer Modeling           %%
%% Mathematical Biosciences Institute, Sept. 7-10, 2010 %%
%% K.A. Rejniak, Moffitt Cancer Center & Research Inst. %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


disp(' ________________________________________________________');
disp('|                                                        |');
disp('| This simulations will show development of a 3D cluster |');
disp('| of tumor cells that can either migrate or grow.        |');
disp('| The probability of initiating  each process by a cell, |');
disp('| cell migration speed,  and cell doubling time  are all |');
disp('| defined by the user.                                   |');
disp('|________________________________________________________|');
disp(' ');
disp(' ');



%% cell diam=10 microns; domain=150cells=1.5mm
%% base doubling time=6h;  
DomNum=150;       % number of sites in each direction; #cells
IterNum=50;       % number of iterations      

domain=zeros(DomNum,DomNum,DomNum);  % 3D empty domain
tumor=zeros(DomNum*DomNum*DomNum,4); % tumor cells: [x,y,z,age]
DivSid=[-1,0,0;1,0,0;0,-1,0;0,1,0;0,0,-1;0,0,1]; % 6 dir for cell division  
DivDir=6;
MovSid=[-1,-1,-1;-1,-1, 0;-1,-1, 1;-1, 0,-1;-1, 0, 0;-1, 0, 1;-1, 1,-1;...
        -1, 1, 0;-1, 1  1; 0,-1,-1; 0,-1, 0; 0,-1, 1; 0, 0,-1; 0, 0, 1;...
         0, 1,-1; 0, 1, 0; 0, 1, 1; 1,-1,-1; 1,-1, 0; 1,-1, 1; 1, 0,-1;...
         1, 0, 0; 1, 0, 1; 1, 1,-1; 1, 1, 0; 1, 1, 1]; % 26 div for cell 
MovDir=26;                                             % migration


DoubTime=DefineDoublingTime;
MoveSped=DefineMovementSpeed;
MoveProb=DefineMovementProbability;


Ntumor=1;                            % current number of tumor cells
tumor(Ntumor,1:4)=[DomNum/2,DomNum/2,DomNum/2,1]; % first cell in the middle
domain(DomNum/2,DomNum/2,DomNum/2)=1; % middle site is filled  


scrz=get(0,'ScreenSize');
figure('Position',[10,scrz(4)/3,2*scrz(3)/3,2*scrz(4)/3])
DrawTumor(tumor,Ntumor,DomNum,0)



%% iterations %%
disp(' ... 3D tumor cluster is growing ...');
for Niter=1:IterNum
  tumor(:,4)=tumor(:,4)+1;     % increase cell age
  for jj=1:Ntumor              % inspect all tumor cells
 
    if (MoveProb==0) 
      chance=100;             % no migration
    elseif (MoveProb==100)
      chance=0;             % sure migration
    else  
      chance=randi(100);    % probability
    end
    
    if (chance<MoveProb)  %% cell is moving if there is a space
      tumxy=[tumor(jj,1),tumor(jj,2),tumor(jj,3)];     
      Mchoise=zeros(MovDir,1);
      kch=0;
      for kk=1:MovDir 
        movxy=MoveSped*[MovSid(kk,1),MovSid(kk,2),MovSid(kk,3)];  
        if ((tumxy(1)+movxy(1)>0)&&(tumxy(1)+movxy(1)<=DomNum)&&...
          (tumxy(2)+movxy(2)>0)&&(tumxy(2)+movxy(2)<=DomNum)&&(tumxy(3)+...
          movxy(3)>0)&&(tumxy(3)+movxy(3)<=DomNum))           
          if (domain(tumxy(1)+movxy(1),tumxy(2)+movxy(2),tumxy(3)+movxy(3))==0)
            kch=kch+1;
            Mchoice(kch)=kk;
          end
        end
      end  
      if (kch>1)
        which=randi(kch);
        domain(tumxy(1),tumxy(2),tumxy(3))=0;
        movxy=MoveSped*[MovSid(Mchoice(which),1),MovSid(Mchoice(which),2),...
          MovSid(Mchoice(which),3)]; 
        tumor(jj,1)=tumxy(1)+movxy(1); % new cell coordinates 
        tumor(jj,2)=tumxy(2)+movxy(2);
        tumor(jj,3)=tumxy(3)+movxy(3);
        domain(tumxy(1)+movxy(1),tumxy(2)+movxy(2),tumxy(3)+movxy(3))=1; % fill a domain site
      end     
    else   % cell is growing if there is a space     
      if (tumor(jj,4)>=DoubTime) % cell must be mature to divide 
        tumxy=[tumor(jj,1),tumor(jj,2),tumor(jj,3)];     
        Dchoice=zeros(DivDir,1); 
        kch=0;
        for kk=1:DivDir  
          if ((tumxy(1)+DivSid(kk,1)>0)&&(tumxy(1)+DivSid(kk,1)<=DomNum)...
            &&(tumxy(2)+DivSid(kk,2)>0)&&(tumxy(2)+DivSid(kk,2)<=DomNum)...
            &&(tumxy(3)+DivSid(kk,3)>0)&&(tumxy(3)+DivSid(kk,3)<=DomNum))  
            if (domain(tumxy(1)+DivSid(kk,1),tumxy(2)+DivSid(kk,2),...
              tumxy(3)+DivSid(kk,3))==0)
              kch=kch+1;
              Dchoice(kch)=kk;
            end
          end
        end
        if (kch>1)
          tumor(jj,4)=0;         % mother cell become a doughter cell  
                 
          which=randi(kch);
          Ntumor=Ntumor+1;           % add a new tumor cell
          tumor(Ntumor,1)=tumxy(1)+DivSid(Dchoice(which),1); % new cell  
          tumor(Ntumor,2)=tumxy(2)+DivSid(Dchoice(which),2); % coordinates 
          tumor(Ntumor,3)=tumxy(3)+DivSid(Dchoice(which),3);
          tumor(Ntumor,4)=0;
          domain(tumxy(1)+DivSid(Dchoice(which),1),tumxy(2)+... % fill a domain site
            DivSid(Dchoice(which),2),tumxy(3)+DivSid(Dchoice(which),3))=1; 
        end
      end
    end
   
  end
    
  % draw a new cell
  if (mod(Niter,3)==0)
    DrawTumor(tumor,Ntumor,DomNum,Niter)
  end
end

DrawTumor(tumor,Ntumor,DomNum,Niter)
RotateTumor(tumor,Ntumor,DomNum,Niter)

DrawZsections(tumor,Ntumor,DomNum,Niter)

DrawTumorAge(tumor,Ntumor,DomNum,0.5)


fname=['Ex6_',num2str(DoubTime*6),'h_',num2str(MoveSped),'sp_',num2str(floor(MoveProb)),'perc'];
print('-djpeg100',fname)


disp(' ');
disp('The End!');
disp(' ');
clear all

end



%%%%%%%%%%%%%%%%%%%%%%
function outVal=DefineDoublingTime
  cond=1;
  while (cond>0)
    dt=input('choose doubling time between 6 and 36 hours: ');
    if ((dt>=6)&&(dt<=36)) cond=0; end
  end
  outVal=dt/6;
  disp('Thank you!')
  disp(' ')
end

%%%%%%%%%%%%%%%%%%%%%%
function outVal=DefineMovementSpeed
  cond=1;
  while (cond>0)
    dt=input('choose movement speed: 1, 2 or 3 (x1.5 microns per h): ');
    %%% unit is 10 micron per 6 hours, i.e., 0.25x10^-5 mm/min
    if ((dt>=1)&&(dt<=3)) cond=0; end
  end
  outVal=round(dt);
  disp('Thank you!')
  disp(' ')
end

%%%%%%%%%%%%%%%%%%%%%%
function outVal=DefineMovementProbability
  cond=1;
  while (cond>0)
    dt=input('choose probabilty [0..100] of cell migration over cell grow: ');
    if ((dt>=0)&&(dt<=100)) cond=0; end
  end
  outVal=dt;
  disp('Thank you!')
  disp(' ')
end

%%%%%%%%%%%%%%%%%%%%%%
function DrawTumor(TumVect,TumNum,DomNum,iter)
 
  clf
  p1=subplot('Position',[0.0,0.2,0.65,0.65]);
  axis([-1,DomNum+1,-1,DomNum+1,-1,DomNum+1])
  hold on
  axis equal
  axis([-1,DomNum+1,-1,DomNum+1,-1,DomNum+1])
  grid
    
  plot3(TumVect(1:TumNum,1),TumVect(1:TumNum,2),TumVect(1:TumNum,3),...
      'ko','MarkerFaceColor','g')
  title(['number of cells: ',num2str(TumNum),';  final time: ',...
      num2str(iter*6),' hrs'])
  pause(0.05)
end

%%%%%%%%%%%%%%%%%%%%%%
function RotateTumor(TumVect,TumNum,DomNum,iter)
  disp('... the final tumor cluster configuration will be rotated ...');  
    
  for ii=1:19 
    view(-37.5+10*ii,30); 
    pause(0.1); 
  end
end

%%%%%%%%%%%%%%%%%%%%%%
function DrawZsections(TumVect,TumNum,DomNum,iter)
  numZ=7; numZ2=floor(numZ/2);
  Zcol=['k','y','b','r','g','m','c'];
  
  minSl=min(TumVect(1:TumNum,3));
  maxSl=max(TumVect(1:TumNum,3));
  numSl=1+floor((maxSl-minSl)/numZ);
  
  if (numSl<=1)
    disp(['tumor too small to plot ',num2str(numZ),' z-sections']);  
  else
    disp('... z-sections through the tumor cluster will be drawn...');  
 
    minDx=min(TumVect(1:TumNum,1));
    maxDx=max(TumVect(1:TumNum,1));
    minDy=min(TumVect(1:TumNum,2));
    maxDy=max(TumVect(1:TumNum,2));
    minDz=min(TumVect(1:TumNum,3));
    maxDz=max(TumVect(1:TumNum,3));
    
    
    p2=subplot('Position',[0.5,0.2,0.5,0.5]);
    axis([minSl,maxSl,minSl,maxSl,minSl,maxSl])
    hold on
    axis equal
    axis([minSl,maxSl,minSl,maxSl,minSl,maxSl])
    grid
    
    title([num2str(numZ),'   z-sections through the tumor cluster'])

    tumZcoo=TumVect(1:TumNum,3);

    minZ=(DomNum/2)-numZ2*numSl;
    maxZ=(DomNum/2)+numZ2*numSl;
    kk=0;
    for indx=minZ:numSl:maxZ  
      kk=kk+1;  
     
      Ind=find(tumZcoo==indx);
      sInd=size(Ind,1);
      plot3(TumVect(Ind(1:sInd),1),TumVect(Ind(1:sInd),2),...
        TumVect(Ind(1:sInd),3),'ko','MarkerFaceColor',Zcol(kk))
      pause(0.1);
    end    
  end
end

%%%%%%%%%%%%%%%%%%%%%%
function DrawTumorAge(TumVect,TumNum,DomNum,mode)
  disp('... tumor age structure will be drawn ...');  
  
  minDx=min(TumVect(1:TumNum,1));
  maxDx=max(TumVect(1:TumNum,1));
  minDy=min(TumVect(1:TumNum,2));
  maxDy=max(TumVect(1:TumNum,2));
  minDz=min(TumVect(1:TumNum,3));
  maxDz=max(TumVect(1:TumNum,3));
    
  p3=subplot('Position',[0.0,0.2,0.5,0.5]);
  axis([minDx,maxDx,minDy,maxDy,minDz,maxDz])
  hold on
  axis equal
  axis([minDx,maxDx,minDy,maxDy,minDz,maxDz])
  grid
  
  numAge=7;
  minAge=min(TumVect(1:TumNum,4));
  maxAge=max(TumVect(1:TumNum,4));
  stepAge=1+floor((maxAge-minAge)/numAge);
  Acol=['g','b','r','y','m','c','k'];
 
  
  %%% draws the cell age structure drawing each age group separately 
  if (mode==1)
    tumAge=TumVect(1:TumNum,4);
    tumAge=1+floor(tumAge/stepAge);
    for kk=numAge:-1:1
      Ind=find(tumAge==kk);
      sInd=size(Ind,1);
      plot3(TumVect(Ind(1:sInd),1),TumVect(Ind(1:sInd),2),...
        TumVect(Ind(1:sInd),3),'o','MarkerEdgeColor','k',...
        'MarkerFaceColor',Acol(kk))   
      pause(0.5)
      hold on
    end
  end
     
  
  %%% draws a cluster with one-quarter to show inside age structure  
  if (mode==0.5)
    DomNum2=DomNum/2;
    NewTumNum=0;
    NewTumVect=zeros(size(TumVect)); 
    for jj=1:TumNum
      if ((TumVect(jj,2)<DomNum2)&&(TumVect(jj,3)>DomNum2))
      else
        NewTumNum=NewTumNum+1;
        NewTumVect(NewTumNum,1:4)=TumVect(jj,1:4);
      end
    end
   
    tumAge=NewTumVect(1:NewTumNum,4);
    tumAge=1+floor(tumAge/stepAge);
    for kk=numAge:-1:1
      Ind=find(tumAge==kk);     
      sInd=size(Ind,1);
      plot3(NewTumVect(Ind(1:sInd),1),NewTumVect(Ind(1:sInd),2),...
        NewTumVect(Ind(1:sInd),3),'o','MarkerEdgeColor',Acol(kk),...
        'MarkerFaceColor',Acol(kk))
      pause(0.5)
    end
    title('color-coded age structure of a 3D tumor cluster')
  end
   
end

%%%%%%%%%%%%%%%%%%%%%%
