function CellColonyDrug

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  PASI Workshop, Porto Alegre, Brazil, July 20 & 22, 2013   %%
%%  Kasia A. Rejniak - Moffitt Cancer Research Institute, IMO %%
%%  Drug resistance in tumor cells, cell colony model (2)     %%
%%  Applies drug to eradicate the tumor cell colony           %% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

disp('this function will apply drug to eradicate tumor cluster')
disp('********************************************************')
disp(' ')



% parameter to control overall toxicity and cell response to the drug
TotalTox=1500;            % total toxicity in the dish
cellDose=   0.15;          % drug absroption by each cell in each iteration
cellThrs=   7.5;          % cell toxicity death treshold
deathProb=  0.75;         % probability of cell dying if toxic 




% predefined initial data 
cells=load('CellCluster.txt'); % start with the already formed cell colony
Ncells=size(cells,1);          % number of cells
                               % cells [x,y,age,mutation,drug]


% predefined parameters
param=load('CellClusterParameters.txt');  % read model parameters
AgeDiv=param(1);    % age at which the cell will divide [24 hours]
divChance=param(2); % chance of division in current iteration
diam=param(3);      % cell diameter [5 microns]
epsDiam=param(4);   % distance for placing a new cell [2.5 microns]
dt=param(5);        % time step [half an hour = 30 min]
Spr=param(6);       % spring stiffness
nu=param(7);        % mass/viscosity term
xmin=param(8); xmax=param(9); ymin=param(10); ymax=param(11);




disp('******************************')
disp(' possible treatment schedule: ')
disp('     1. bi-weekly  ')
disp('     2. daily      ')
disp('     3. twice daily')
disp('     4. hourly     ')
disp('     5. continuous ')
disp('')
option=input('choose option: ');
if (option==1)
  DoseIter=round(0.5*7*24/dt);   % twice per week
elseif (option==2)
  DoseIter=round(24/dt);         % once per day
elseif (option==3)
  DoseIter=round(12/dt);         % twice per day
elseif (option==4)
  DoseIter=round(1/dt);          % once per hour
elseif (option==5)
  DoseIter=1;                    % once per iteration
else
  disp('chosen number is out of range -- no treatment')  
  DoseIter=0; pause
end %if
disp(' ')


disp(['maximal tolerated total concentration of drug is ',num2str(TotalTox)])
DoseTox=input('specify the amount of drug added to the dish in each treatment: ');
disp(' ')


MutRate=input('spacify the percentage of mutated cells [0-10]: ');
if ((MutRate<0) || (MutRate>10))
  disp('chosen number is out of range')
  MutRate=0; pause
end %if
disp(' ')





% number of iterations and variales that store the whole treatment results
Niter=400;  % number of iterations
SimCel=zeros(Niter,1);   % remeber number of cells at each iteration 
SimTox=zeros(Niter,1);   % remeber cumulative toxicity at each iteration


% cell mutations
Mrate=floor(MutRate*Ncells/100);        % mutation rate
Mind=mod(floor(10000*rand(Mrate,1)),Ncells); % mutation index
for ii=1:size(Mind) cells(Mind(ii),4)=1; end 





%%%%%%%%%%%%%%%%%
% draw the cells
fg=figure('position',[150,150,1000,1000]);

subplot('position',[0.25,0.45,0.5,0.5])
axis([xmin,xmax,ymin,ymax])
axis equal
axis([xmin,xmax,ymin,ymax])
hold on

subplot('position',[0.1,0.25,0.8,0.15])
axis([0,Niter,0,1.25*TotalTox]) 
hold on
 
subplot('position',[0.1,0.05,0.8,0.15])
axis([0,Niter,0,1500])
hold on
pause(0.1)






%%%% iterations
iter=0;
DrugMedium=DoseTox;      % initial amount of drug in the medium 
while ((iter<Niter)&&(Ncells>0))  % finish if Niter or if no more cells
  iter=iter+1;
    
  if (mod(iter,DoseIter)==0)      % add drug to the medium
    DrugMedium=DrugMedium+DoseTox;
  end %if
  SimTox(iter,1)=DrugMedium;      % remember total toxicity
  
  
  % cells get older, absorb the drug & may increase its resistance to drug 
  % if mutated
  for ii=1:Ncells
    cells(ii,3)=cells(ii,3)+dt;     % cell gets older
    
    if (cells(ii,4)>0) && (mod(iter,150)==0)
      cells(ii,4)=cells(ii,4)+1;  % increase in cell resistance  
    end %if
    
    if (DrugMedium>=cellDose)   % absorb the drug from medium if availble 
      cells(ii,5)=cells(ii,5)+cellDose;
      DrugMedium=DrugMedium-cellDose; % reduce the drug in medium
    end %if
  end %for
  
  
  % cells will die if accumulated drug is larger than threshold 
  % after the cell dies, other cells need to be rearranged
  cells_memo=cells;  % for rearrangement of living cells 
  Ncells_memo=Ncells; 
  clear cells    
  
  Ncells=0; cells=zeros(1,2); 
  for ii=1:Ncells_memo
    if (cells_memo(ii,5)<(cells_memo(ii,4)+cellThrs))     
      Ncells=Ncells+1;            % remember the cell if alive
      cells(Ncells,1:5)=cells_memo(ii,1:5);
    else                          % if not, kill the cell with deathProb 
      chance=rand(1);
      if (chance <(1-deathProb))  % cell will not be killed
        Ncells=Ncells+1;
        cells(Ncells,1:5)=cells_memo(ii,1:5);
      end %if
    end %if   
  end %for
  clear cells_memo Ncells_memo
  
  
  % cells will divide with prob divChance if they are old enough
  % mutated cells will divide with prob 1
  for ii=1:Ncells   % inspect all cells
    chance=rand(1);            % compute a random number  
    if (cells(ii,3)>=AgeDiv)
     if (cells(ii,4)>0) || (chance<divChance)  
      cells(ii,3)=0;           % age of the first daughter cell set to 0
      cells(ii,5)=0.5*cells(ii,5); % drug concentration is set to half
                               % x,y position & mutation the same as the mother cell
                                          
      Ncells=Ncells+1;         % second daughter cell 
      cells(Ncells,3)=0;       % age of the second daughter cell set to 0    
      cells(Ncells,4)=cells(ii,4); % inherited mutation status
      cells(Ncells,5)=cells(ii,5); % drug concentration is set to half
      ang=randi(62);           % random angle to place a new cell
      cells(Ncells,1:2)=[cells(ii,1)+epsDiam*cos(2*pi/(ang*0.1)),...
                         cells(ii,2)+epsDiam*sin(2*pi/(ang*0.1))];      
     end   % if
    end % if
  end  % for  
  
  % push cells apart if too close
  force=DefineForce(cells,diam,Spr);  % find forces between the cells  
  for ii=1:Ncells  % inpect all cells
    cells(ii,1)=cells(ii,1)+(force(ii,1)*dt/nu);  % new cell positions due
    cells(ii,2)=cells(ii,2)+(force(ii,2)*dt/nu);  % to repulsive forces 
  end %for
  SimCel(iter,1)=Ncells; % remember the number of cells in each simulation

  
  
  
  
  %%%%%%%%%%%%%%%%%
  % draw the cells
  clf
    
  subplot('position',[0.25,0.45,0.5,0.5])
  axis([xmin,xmax,ymin,ymax])
  axis equal
  axis([xmin,xmax,ymin,ymax])
  hold on
  % draw cells -- differnet shades of blue for levels of absorbed drug
  for ii=1:Ncells
    if (cells(ii,4)>0)
      scale=min(1,cells(ii,5)/(cells(ii,4)+cellThrs));  
      plot(cells(ii,1),cells(ii,2),'ro','MarkerFaceColor',[0,1-scale,1],...
          'MarkerSize',7)     % if mutated has red perimeter
    else
      scale=min(1,cells(ii,5)/cellThrs);  
      plot(cells(ii,1),cells(ii,2),'ko','MarkerFaceColor',[0,1-scale,1],...
          'MarkerSize',7)     % if not mutated has black perimeter     
    end %if
  end %for 
  
  text(-65,-65,'drug: ')
  for jj=1:10 plot(-50+10*jj,-65,'o','MarkerFaceColor',[0,1-(jj-1)*0.1,1]); end
  title(['dose: ',num2str(DoseTox),',  # cells:',num2str(Ncells),...
      ',  total drug:',num2str(DrugMedium,2)],'FontSize',17)
  
  
  
  subplot('position',[0.1,0.25,0.8,0.15])
  axis([0,Niter,0,1.25*TotalTox]) % maximal toxicity
  hold on
  plot(1:1:iter,SimTox(1:1:iter,1),'r*')
  ylabel('total toxicity','Fontsize',13)
  plot([0,Niter],[TotalTox,TotalTox],'r')
   
  MaxCell=1500;
  subplot('position',[0.1,0.05,0.8,0.15])
  axis([0,Niter,0,MaxCell])  % maximal number of cells
  hold on
  plot(1:1:iter,SimCel(1:1:iter,1),'b*')
  xlabel('iterations [0.5*hour]','FontSize',13)
  ylabel('number of cells','FontSize',13)
  tt=0:round(24/dt):Niter;
  for ii=1:size(tt,2) plot([tt(ii),tt(ii)],[0,MaxCell],'r'); end
  
  pause(0.1)
  
end %while
  
clear all
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function defining the forces between all neighboring cells         % 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function force=DefineForce(cells,dist,Spr) 
  Csize=size(cells,1);
  force=zeros(Csize,2);
  if (Csize>=3) 
    for ii=1:Csize-1
      for jj=ii+1:Csize  
        dx=cells(ii,1)-cells(jj,1);
        dy=cells(ii,2)-cells(jj,2);
        dxy=sqrt(dx*dx+dy*dy);
        
        force(ii,1:2)=force(ii,1:2)+Spr*max((dist-dxy),0)*[dx,dy];  
        force(jj,1:2)=force(jj,1:2)-Spr*max((dist-dxy),0)*[dx,dy];  
      end
    end
  end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function defining the forces between all neighboring cells         %
% using the Delaunay triangultion                                    % 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function force=DefineForceDT(cells,dist,Spr) 
  Csize=size(cells,1);
  force=zeros(Csize,2);
  if (Csize>=3) 
    neigh=delaunay(cells(:,1),cells(:,2)); % find neighbors
    Nneigh=size(neigh,1);
    
    for ii=1:Nneigh % inspect all triangles
      num1=neigh(ii,1); num2=neigh(ii,2); num3=neigh(ii,3);  

      dx=cells(num1,1)-cells(num2,1);
      dy=cells(num1,2)-cells(num2,2);
      dxy=sqrt(dx*dx+dy*dy);
      force(num1,1:2)=force(num1,1:2)+Spr*max((dist-dxy),0)*[dx,dy];  
      force(num2,1:2)=force(num2,1:2)-Spr*max((dist-dxy),0)*[dx,dy];  
       
      dx=cells(num1,1)-cells(num3,1);
      dy=cells(num1,2)-cells(num3,2);
      dxy=sqrt(dx*dx+dy*dy);
      force(num1,1:2)=force(num1,1:2)+Spr*max((dist-dxy),0)*[dx,dy];  
      force(num3,1:2)=force(num3,1:2)-Spr*max((dist-dxy),0)*[dx,dy];  

      dx=cells(num3,1)-cells(num2,1);
      dy=cells(num3,2)-cells(num2,2);
      dxy=sqrt(dx*dx+dy*dy);
      force(num3,1:2)=force(num3,1:2)+Spr*max((dist-dxy),0)*[dx,dy];  
      force(num2,1:2)=force(num2,1:2)-Spr*max((dist-dxy),0)*[dx,dy];      
    end
  end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



